// $Id: robot.h,v 1.7 2002/03/27 02:28:28 ccf7f Exp $
// Flancrest Enterprises, Group 22
// About this file:
// This class hides design decisions involving how information is sent to
// and received from the robot server (which hides how the actual hardware
// interfaces). In fact, this class completely abstracts the functional
// aspects of there existing a robot server.

#ifndef ROBOT_H
#define ROBOT_H

#include <string>
#include <afxmt.h>
#include "cmd.h"
#include "ws-util.h"
using namespace std;

class CFangDlg;

typedef DWORD timeSpan;    // milliseconds
typedef DWORD currentTime; // milliseconds

const bool debugPrintsCmds = true;

class Robot {
// Associations
private:
	CFangDlg *gui;

// Attributes
private:
	string hostname;
	WSAData wsaData;
	SOCKET sd;
	const int port;
	
	// Use data member access functions, touching these directly
	// is not thread safe.
	bool connected;
	currentTime lastHeartBeat;
	timeSpan heartBeatRate;
	bool haveHeartBeat;

	// Accessed only through respective mutexes
	// (used for inter-thread communication)
	string cmdToSend;
	string cmdRecvd;

	CEvent sendEvent, sendEventDoneEvent, recvEvent, recvEventDoneEvent,
		netRecvEventDoneEvent,
		heartBeatCheckEvent, heartBeatCheckEventDoneEvent,
		sendHeartBeatEvent, sendHeartBeatEventDoneEvent,
		ifaceEvent, ifaceEventDoneEvent;
	CMutex cmdToSendMutex, cmdRecvdMutex, lastHeartBeatMutex,
		heartBeatRateMutex, connectedMutex, haveHeartBeatMutex, guiMutex;
	
	const string className;
	const string cmdBeginSymbol;
	const string cmdDelimSymbol;
	const string cmdEndSymbol;

// Operations
public:
	// Sets the private date member *gui to the argument. Returns true
	// if gui was successfully stored.
	bool setGui(CFangDlg *gui);
	Robot();
	~Robot();
	// Connects to the given robot server host. Returns true if it was able
	// to connect.
	bool connect(const string hostname);
	// Disconnects from the robot server, returns true it was able to disconnect
	bool disconnect();
	// Tells the robot to do command, returns true if it was able to send the
	// message to the robot.
	bool tell(const Cmd command);

	// These allow threads in member functions of this class to be started.
	// param is a pointer to this class instance's memory address.
	// Returns 0 on success.
	friend UINT startNetSend(LPVOID param);
	friend UINT startNetRecv(LPVOID param);
	friend UINT startIface(LPVOID param);
	friend UINT startHeartBeatCheck(LPVOID param);
	friend UINT startSendHeartBeat(LPVOID param);
private:
	//
	// These are the threads of Robot.
	//
	// Sends data to the network
	UINT netSend();
	// Receives data from the network
	UINT netRecv();
	// Processes data received from the network
	UINT iface();
	// Checks that we are receiving a heartBeat from the robot server
	UINT heartBeatCheck();
	// Sends a heartBeat to the robot server
	UINT sendHeartBeat();

	//
	// Access functions for non-thread safe private data members
	//
	// get the heartBeatRate
	timeSpan getHeartBeatRate();
	// sets the heartbeat rate to newHeartBeatRate
	void  setHeartBeatRate(const timeSpan newHeartBeatRate);
	// gets the time of the last heartBeat received
	currentTime getLastHeartBeat();
	// sets the tiem of the last heartBeat received
	void        setLastHeartBeat(const currentTime newLastHeartBeat);
	// returns true if we have a heartBeat from the robot server
	bool getHaveHeartBeat();
	// sets whether or not we have a heartBeat from the server
	void setHaveHeartBeat(const bool haveIt);
	// returns true if we are currently connected to the robot server
	bool getConnected();
	// sets whether or not we are currently connected to the robot server
	void setConnected(const bool newConnected);

	// Parses command and turns it into a Cmd. Returns Cmd("", emptyVector)
	// if the command is invalid.
	Cmd string2Cmd(const string command) const;
	// Turns command into the format used to send the command over the network
	string Cmd2String(const Cmd command) const;
	
	// Passes cmd onto the gui
	void guiUpdate(const Cmd cmd);

	// Initializes variables
	void initVariables();
	// Sends the "init" command to the robot server
	void sendInitCmd();
	
	// if pcHost is a hostname, returns the ip address. if pcHost is an
	// ip address. returns pcHost
	u_long lookupAddress(const string pcHost) const;
	// brings up the connection to the remoteAddr (an ip addr) on the port
	// port. Returns the socket.
	SOCKET establishConnection(const u_long remoteAddr,const u_short port)const;
};

#endif
